'use strict';

/**
 * Base class for API errors. Contains indication of HTTP status.
 */

Object.defineProperty(exports, "__esModule", {
  value: true
});
class ApiError extends Error {

  /**
   * ApiError constructor
   * @param {Function} clazz error name
   * @param {Object} message error message
   * @param {number} status HTTP status
   */
  constructor(clazz, message, status) {
    super(message);
    /**
     * Error name
     * @type {string}
     */
    this.name = clazz.name;
    /**
     * HTTP status code
     * @type {number}
     */
    this.status = status;
    Error.captureStackTrace(this, clazz);
  }

  /**
   * Sets error code, used for i18n
   * @param {string} code error code for i18n
   */
  set code(code) {
    this._code = code;
  }

  /**
   * Returns error code used for i18n
   * @return {string} error code
   */
  get code() {
    return this._code;
  }

  /**
   * Set message arguments for i18n
   * @param {Array<Object>} args arguments for i18n
   */
  set arguments(args) {
    this._args = args;
  }

  /**
   * Returns message arguments for i18n
   * @return {Array<Object>} message arguments for i18n
   */
  get arguments() {
    return this._args;
  }

}

exports.ApiError = ApiError; /**
                              * Throwing this error results in 404 (Not Found) HTTP response code.
                              */

class NotFoundError extends ApiError {

  /**
   * Represents NotFoundError.
   * @param {string} message error message
   */
  constructor(message) {
    super(NotFoundError, message, 404);
  }

}

exports.NotFoundError = NotFoundError; /**
                                        * Throwing this error results in 403 (Forbidden) HTTP response code.
                                        */

class ForbiddenError extends ApiError {

  /**
   * Constructs forbidden error.
   * @param {string} message error message
   */
  constructor(message) {
    super(ForbiddenError, message, 403);
  }

}

exports.ForbiddenError = ForbiddenError; /**
                                          * Throwing this error results in 401 (Unauthorized) HTTP response code.
                                          */

class UnauthorizedError extends ApiError {

  /**
   * Constructs unauthorized error.
   * @param {string} message error message
   */
  constructor(message) {
    super(UnauthorizedError, message, 401);
  }

}

exports.UnauthorizedError = UnauthorizedError; /**
                                                * Represents validation error. Throwing this error results in 400 (Bad Request) HTTP response code.
                                                */

class ValidationError extends ApiError {

  /**
   * Constructs validation error.
   * @param {string} message error message
   * @param {Object} details error data
   */
  constructor(message, details) {
    super(ValidationError, message, 400);
    /**
     * Validation error details
     * @type {Object}
     */
    this.details = details;
  }

}

exports.ValidationError = ValidationError; /**
                                            * Represents unexpected error. Throwing this error results in 500 (Internal Error) HTTP response code.
                                            */

class InternalError extends ApiError {

  /**
   * Constructs unexpected error.
   * @param {string} message error message
   */
  constructor(message) {
    super(InternalError, message, 500);
  }

}

exports.InternalError = InternalError; /**
                                        * Represents conflict error. Throwing this error results in 409 (Conflict) HTTP response code.
                                        */

class ConflictError extends ApiError {
  /**
   * Constructs conflict error.
   * @param {string} message error message
   */
  constructor(message) {
    super(ConflictError, message, 409);
  }
}

exports.ConflictError = ConflictError; /**
                                        * @typedef {Object} TooManyRequestsErrorMetadata
                                        * @property {Number} periodInMinutes throttling period in minutes
                                        * @property {Number} requestsPerPeriodAllowed available requests for periodInMinutes
                                        * @property {Date} recommendedRetryTime recommended date to retry request
                                        */

/**
 * Represents too many requests error. Throwing this error results in 429 (Too Many Requests) HTTP response code.
 */

class TooManyRequestsError extends ApiError {
  /**
   * Constructs too many requests error.
   * @param {string} message error message
   * @param {TooManyRequestsErrorMetadata} metadata error metadata
   */
  constructor(message, metadata) {
    super(TooManyRequestsError, message, 429);
    this.metadata = metadata;
  }
}
exports.TooManyRequestsError = TooManyRequestsError;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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