'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _promise = require('babel-runtime/core-js/promise');

var _promise2 = _interopRequireDefault(_promise);

var _keys = require('babel-runtime/core-js/object/keys');

var _keys2 = _interopRequireDefault(_keys);

var _logger = require('../../logger');

var _logger2 = _interopRequireDefault(_logger);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Subscription manager to handle account subscription logic
 */
class SubscriptionManager {

  /**
   * Constructs the subscription manager
   * @param {MetaApiWebsocketClient} websocketClient websocket client to use for sending requests
   */
  constructor(websocketClient) {
    this._websocketClient = websocketClient;
    this._subscriptions = {};
    this._awaitingResubscribe = {};
    this._subscriptionState = {};
    this._logger = _logger2.default.getLogger('SubscriptionManager');
  }

  /**
   * Returns whether an account is currently subscribing
   * @param {String} accountId account id
   * @param {Number} instanceNumber instance index number
   * @returns {Boolean} whether an account is currently subscribing
   */
  isAccountSubscribing(accountId, instanceNumber) {
    if (instanceNumber !== undefined) {
      return (0, _keys2.default)(this._subscriptions).includes(accountId + ':' + instanceNumber);
    } else {
      for (let key of (0, _keys2.default)(this._subscriptions)) {
        if (key.startsWith(accountId)) {
          return true;
        }
      }
      return false;
    }
  }

  /**
   * Returns whether an instance is in disconnected retry mode
   * @param {String} accountId account id
   * @param {Number} instanceNumber instance index number
   * @returns {Boolean} whether an account is currently subscribing
   */
  isDisconnectedRetryMode(accountId, instanceNumber) {
    let instanceId = accountId + ':' + (instanceNumber || 0);
    return this._subscriptions[instanceId] ? this._subscriptions[instanceId].isDisconnectedRetryMode : false;
  }

  /**
   * Returns whether an account subscription is active
   * @param {String} accountId account id
   * @returns {Boolean} instance actual subscribe state
   */
  isSubscriptionActive(accountId) {
    return !!this._subscriptionState[accountId];
  }

  /**
   * Subscribes to the Metatrader terminal events
   * @param {String} accountId id of the MetaTrader account to subscribe to
   * @param {Number} [instanceNumber] instance index number
   * @returns {Promise} promise which resolves when subscription started
   */
  subscribe(accountId, instanceNumber) {
    this._subscriptionState[accountId] = true;
    return this._websocketClient.rpcRequest(accountId, { type: 'subscribe', instanceIndex: instanceNumber });
  }

  /**
   * Schedules to send subscribe requests to an account until cancelled
   * @param {String} accountId id of the MetaTrader account
   * @param {Number} instanceNumber instance index number
   * @param {Boolean} isDisconnectedRetryMode whether to start subscription in disconnected retry
   * mode. Subscription task in disconnected mode will be immediately replaced when the status packet is received
   */
  async scheduleSubscribe(accountId, instanceNumber, isDisconnectedRetryMode = false) {
    const client = this._websocketClient;
    let instanceId = accountId + ':' + (instanceNumber || 0);
    if (!this._subscriptions[instanceId]) {
      this._subscriptions[instanceId] = {
        shouldRetry: true,
        task: null,
        waitTask: null,
        future: null,
        isDisconnectedRetryMode
      };
      let subscribeRetryIntervalInSeconds = 3;
      while (this._subscriptions[instanceId].shouldRetry) {
        let resolveSubscribe;
        this._subscriptions[instanceId].task = { promise: new _promise2.default(res => {
            resolveSubscribe = res;
          }) };
        this._subscriptions[instanceId].task.resolve = resolveSubscribe;
        // eslint-disable-next-line no-inner-declarations
        let subscribeTask = async () => {
          try {
            await this.subscribe(accountId, instanceNumber);
          } catch (err) {
            if (err.name === 'TooManyRequestsError') {
              const socketInstanceIndex = client.socketInstancesByAccounts[accountId];
              if (err.metadata.type === 'LIMIT_ACCOUNT_SUBSCRIPTIONS_PER_USER') {
                this._logger.error(`${instanceId}: Failed to subscribe`, err);
              }
              if (['LIMIT_ACCOUNT_SUBSCRIPTIONS_PER_USER', 'LIMIT_ACCOUNT_SUBSCRIPTIONS_PER_SERVER', 'LIMIT_ACCOUNT_SUBSCRIPTIONS_PER_USER_PER_SERVER'].includes(err.metadata.type)) {
                delete client.socketInstancesByAccounts[accountId];
                client.lockSocketInstance(socketInstanceIndex, err.metadata);
              } else {
                const retryTime = new Date(err.metadata.recommendedRetryTime).getTime();
                if (Date.now() + subscribeRetryIntervalInSeconds * 1000 < retryTime) {
                  await new _promise2.default(res => setTimeout(res, retryTime - Date.now() - subscribeRetryIntervalInSeconds * 1000));
                }
              }
            }
          }
          resolveSubscribe();
        };
        subscribeTask();
        await this._subscriptions[instanceId].task.promise;
        if (!this._subscriptions[instanceId].shouldRetry) {
          break;
        }
        const retryInterval = subscribeRetryIntervalInSeconds;
        subscribeRetryIntervalInSeconds = Math.min(subscribeRetryIntervalInSeconds * 2, 300);
        let resolve;
        let subscribePromise = new _promise2.default(res => {
          resolve = res;
        });
        this._subscriptions[instanceId].waitTask = setTimeout(() => {
          resolve(true);
        }, retryInterval * 1000);
        this._subscriptions[instanceId].future = { resolve, promise: subscribePromise };
        const result = await this._subscriptions[instanceId].future.promise;
        this._subscriptions[instanceId].future = null;
        if (!result) {
          break;
        }
      }
      delete this._subscriptions[instanceId];
    }
  }

  /**
   * Unsubscribe from account
   * @param {String} accountId id of the MetaTrader account to unsubscribe
   * @returns {Promise} promise which resolves when socket unsubscribed
   */
  async unsubscribe(accountId) {
    this.cancelAccount(accountId);
    delete this._subscriptionState[accountId];
    return this._websocketClient.rpcRequest(accountId, { type: 'unsubscribe' });
  }

  /**
   * Cancels active subscription tasks for an instance id
   * @param {String} instanceId instance id to cancel subscription task for
   */
  cancelSubscribe(instanceId) {
    if (this._subscriptions[instanceId]) {
      const subscription = this._subscriptions[instanceId];
      if (subscription.future) {
        subscription.future.resolve(false);
        clearTimeout(subscription.waitTask);
      }
      if (subscription.task) {
        subscription.task.resolve(false);
      }
      subscription.shouldRetry = false;
    }
  }

  /**
   * Cancels active subscription tasks for an account
   * @param {String} accountId account id to cancel subscription tasks for
   */
  cancelAccount(accountId) {
    for (let instanceId of (0, _keys2.default)(this._subscriptions).filter(key => key.startsWith(accountId))) {
      this.cancelSubscribe(instanceId);
    }
  }

  /**
   * Invoked on account timeout.
   * @param {String} accountId id of the MetaTrader account
   * @param {Number} instanceNumber instance index number
   */
  onTimeout(accountId, instanceNumber) {
    if (this._websocketClient.socketInstancesByAccounts[accountId] !== undefined && this._websocketClient.connected(this._websocketClient.socketInstancesByAccounts[accountId])) {
      this.scheduleSubscribe(accountId, instanceNumber, true);
    }
  }

  /**
   * Invoked when connection to MetaTrader terminal terminated
   * @param {String} accountId id of the MetaTrader account
   * @param {Number} instanceNumber instance index number
   */
  async onDisconnected(accountId, instanceNumber) {
    await new _promise2.default(res => setTimeout(res, Math.max(Math.random() * 5, 1) * 1000));
    if (this._websocketClient.socketInstancesByAccounts[accountId] !== undefined) {
      this.scheduleSubscribe(accountId, instanceNumber, true);
    }
  }

  /**
   * Invoked when connection to MetaApi websocket API restored after a disconnect.
   * @param {Number} socketInstanceIndex socket instance index
   * @param {String[]} reconnectAccountIds account ids to reconnect
   */
  onReconnected(socketInstanceIndex, reconnectAccountIds) {
    try {
      const socketInstancesByAccounts = this._websocketClient.socketInstancesByAccounts;
      for (let instanceId of (0, _keys2.default)(this._subscriptions)) {
        const accountId = instanceId.split(':')[0];
        if (socketInstancesByAccounts[accountId] === socketInstanceIndex) {
          this.cancelSubscribe(instanceId);
        }
      }
      reconnectAccountIds.forEach(async accountId => {
        try {
          if (!this._awaitingResubscribe[accountId]) {
            this._awaitingResubscribe[accountId] = true;
            while (this.isAccountSubscribing(accountId)) {
              await new _promise2.default(res => setTimeout(res, 1000));
            }
            delete this._awaitingResubscribe[accountId];
            await new _promise2.default(res => setTimeout(res, Math.random() * 5000));
            this.scheduleSubscribe(accountId);
          }
        } catch (err) {
          this._logger.error(`${accountId}: Account resubscribe task failed`, err);
        }
      });
    } catch (err) {
      this._logger.error('Failed to process subscribe manager reconnected event', err);
    }
  }
}
exports.default = SubscriptionManager;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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