'use strict';

var _promise = require('babel-runtime/core-js/promise');

var _promise2 = _interopRequireDefault(_promise);

var _should = require('should');

var _should2 = _interopRequireDefault(_should);

var _sinon = require('sinon');

var _sinon2 = _interopRequireDefault(_sinon);

var _memoryHistoryStorage = require('./memoryHistoryStorage');

var _memoryHistoryStorage2 = _interopRequireDefault(_memoryHistoryStorage);

var _historyFileManager = require('./historyFileManager');

var _historyFileManager2 = _interopRequireDefault(_historyFileManager);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * @test {MemoryHistoryStorage}
 */
describe('MemoryHistoryStorage', () => {

  let storage;
  let testDeal;
  let testOrder;
  let sandbox;

  before(() => {
    sandbox = _sinon2.default.createSandbox();
  });

  beforeEach(async () => {
    sandbox.stub(_historyFileManager2.default.prototype, 'startUpdateJob').returns();
    storage = new _memoryHistoryStorage2.default('accountId');
    testDeal = { id: '37863643', type: 'DEAL_TYPE_BALANCE', magic: 0, time: new Date(100), commission: 0,
      swap: 0, profit: 10000, platform: 'mt5', comment: 'Demo deposit 1' };
    testOrder = { id: '61210463', type: 'ORDER_TYPE_SELL', state: 'ORDER_STATE_FILLED', symbol: 'AUDNZD', magic: 0,
      time: new Date(50), doneTime: new Date(100), currentPrice: 1, volume: 0.01,
      currentVolume: 0, positionId: '61206630', platform: 'mt5', comment: 'AS_AUDNZD_5YyM6KS7Fv:' };
    await storage.clear();
    storage.onConnected('1:ps-mpa-1', 1);
  });

  afterEach(() => {
    sandbox.restore();
  });

  /**
   * @test {MemoryHistoryStorage#loadDataFromDisk}
   */
  it('should load data from the file manager', async () => {
    sandbox.stub(_historyFileManager2.default.prototype, 'getHistoryFromDisk').returns({ deals: [testDeal], historyOrders: [testOrder] });
    await storage.loadDataFromDisk();
    await new _promise2.default(res => setTimeout(res, 50));
    storage.deals.should.match([testDeal]);
    storage.historyOrders.should.match([testOrder]);
  });

  /**
   * @test {MemoryHistoryStorage#updateDiskStorage}
   */
  it('should update disk storage', async () => {
    sandbox.stub(_historyFileManager2.default.prototype, 'updateDiskStorage');
    await storage.updateDiskStorage();
    _sinon2.default.assert.calledOnce(_historyFileManager2.default.prototype.updateDiskStorage);
  });

  /**
   * @test {MemoryHistoryStorage#lastHistoryOrderTime}
   */
  it('should return last history order time', () => {
    storage.onHistoryOrderAdded('1:ps-mpa-1', {});
    storage.onHistoryOrderAdded('1:ps-mpa-1', { doneTime: new Date('2020-01-01T00:00:00.000Z') });
    storage.onHistoryOrderAdded('1:ps-mpa-1', { doneTime: new Date('2020-01-02T00:00:00.000Z') });
    storage.lastHistoryOrderTime().should.match(new Date('2020-01-02T00:00:00.000Z'));
  });

  /**
   * @test {MemoryHistoryStorage#lastDealTime}
   */
  it('should return last history deal time', () => {
    storage.onDealAdded('1:ps-mpa-1', {});
    storage.onDealAdded('1:ps-mpa-1', { time: new Date('2020-01-01T00:00:00.000Z') });
    storage.onDealAdded('1:ps-mpa-1', { time: new Date('2020-01-02T00:00:00.000Z') });
    storage.lastDealTime().should.match(new Date('2020-01-02T00:00:00.000Z'));
  });

  /**
   * @test {MemoryHistoryStorage#deals}
   */
  it('should return saved deals', () => {
    storage.onDealAdded('1:ps-mpa-1', { id: '1', time: new Date('2020-01-01T00:00:00.000Z'), type: 'DEAL_TYPE_SELL' });
    storage.onDealAdded('1:ps-mpa-1', { id: '7', time: new Date('2020-05-01T00:00:00.000Z'), type: 'DEAL_TYPE_BUY' });
    storage.onDealAdded('1:ps-mpa-1', { id: '8', time: new Date('2020-02-01T00:00:00.000Z'), type: 'DEAL_TYPE_SELL' });
    storage.onDealAdded('1:ps-mpa-1', { id: '6', time: new Date('2020-10-01T00:00:00.000Z'), type: 'DEAL_TYPE_BUY' });
    storage.onDealAdded('1:ps-mpa-1', { id: '4', time: new Date('2020-02-01T00:00:00.000Z'), type: 'DEAL_TYPE_SELL' });
    storage.onDealAdded('1:ps-mpa-1', { id: '5', time: new Date('2020-06-01T00:00:00.000Z'), type: 'DEAL_TYPE_BUY' });
    storage.onDealAdded('1:ps-mpa-1', { id: '11', type: 'DEAL_TYPE_SELL' });
    storage.onDealAdded('1:ps-mpa-1', { id: '3', time: new Date('2020-09-01T00:00:00.000Z'), type: 'DEAL_TYPE_BUY' });
    storage.onDealAdded('1:ps-mpa-1', { id: '5', time: new Date('2020-06-01T00:00:00.000Z'), type: 'DEAL_TYPE_BUY' });
    storage.onDealAdded('1:ps-mpa-1', { id: '2', time: new Date('2020-08-01T00:00:00.000Z'), type: 'DEAL_TYPE_SELL' });
    storage.onDealAdded('1:ps-mpa-1', { id: '10', type: 'DEAL_TYPE_SELL' });
    storage.onDealAdded('1:ps-mpa-1', { id: '12', type: 'DEAL_TYPE_BUY' });
    storage.deals.should.match([{ id: '10', type: 'DEAL_TYPE_SELL' }, { id: '11', type: 'DEAL_TYPE_SELL' }, { id: '12', type: 'DEAL_TYPE_BUY' }, { id: '1', time: new Date('2020-01-01T00:00:00.000Z'), type: 'DEAL_TYPE_SELL' }, { id: '4', time: new Date('2020-02-01T00:00:00.000Z'), type: 'DEAL_TYPE_SELL' }, { id: '8', time: new Date('2020-02-01T00:00:00.000Z'), type: 'DEAL_TYPE_SELL' }, { id: '7', time: new Date('2020-05-01T00:00:00.000Z'), type: 'DEAL_TYPE_BUY' }, { id: '5', time: new Date('2020-06-01T00:00:00.000Z'), type: 'DEAL_TYPE_BUY' }, { id: '2', time: new Date('2020-08-01T00:00:00.000Z'), type: 'DEAL_TYPE_SELL' }, { id: '3', time: new Date('2020-09-01T00:00:00.000Z'), type: 'DEAL_TYPE_BUY' }, { id: '6', time: new Date('2020-10-01T00:00:00.000Z'), type: 'DEAL_TYPE_BUY' }]);
  });

  /**
   * @test {MemoryHistoryStorage#historyOrders}
   */
  it('should return saved historyOrders', () => {
    storage.onHistoryOrderAdded('1:ps-mpa-1', { id: '1', doneTime: new Date('2020-01-01T00:00:00.000Z'),
      type: 'ORDER_TYPE_SELL' });
    storage.onHistoryOrderAdded('1:ps-mpa-1', { id: '7', doneTime: new Date('2020-05-01T00:00:00.000Z'),
      type: 'ORDER_TYPE_BUY' });
    storage.onHistoryOrderAdded('1:ps-mpa-1', { id: '8', doneTime: new Date('2020-02-01T00:00:00.000Z'),
      type: 'ORDER_TYPE_SELL' });
    storage.onHistoryOrderAdded('1:ps-mpa-1', { id: '6', doneTime: new Date('2020-10-01T00:00:00.000Z'),
      type: 'ORDER_TYPE_BUY' });
    storage.onHistoryOrderAdded('1:ps-mpa-1', { id: '4', doneTime: new Date('2020-02-01T00:00:00.000Z'),
      type: 'ORDER_TYPE_SELL' });
    storage.onHistoryOrderAdded('1:ps-mpa-1', { id: '5', doneTime: new Date('2020-06-01T00:00:00.000Z'),
      type: 'ORDER_TYPE_BUY' });
    storage.onHistoryOrderAdded('1:ps-mpa-1', { id: '11', type: 'ORDER_TYPE_SELL' });
    storage.onHistoryOrderAdded('1:ps-mpa-1', { id: '3', doneTime: new Date('2020-09-01T00:00:00.000Z'),
      type: 'ORDER_TYPE_BUY' });
    storage.onHistoryOrderAdded('1:ps-mpa-1', { id: '5', doneTime: new Date('2020-06-01T00:00:00.000Z'),
      type: 'ORDER_TYPE_BUY' });
    storage.onHistoryOrderAdded('1:ps-mpa-1', { id: '2', doneTime: new Date('2020-08-01T00:00:00.000Z'),
      type: 'ORDER_TYPE_SELL' });
    storage.onHistoryOrderAdded('1:ps-mpa-1', { id: '10', type: 'ORDER_TYPE_SELL' });
    storage.onHistoryOrderAdded('1:ps-mpa-1', { id: '12', type: 'ORDER_TYPE_BUY' });
    storage.historyOrders.should.match([{ id: '10', type: 'ORDER_TYPE_SELL' }, { id: '11', type: 'ORDER_TYPE_SELL' }, { id: '12', type: 'ORDER_TYPE_BUY' }, { id: '1', doneTime: new Date('2020-01-01T00:00:00.000Z'), type: 'ORDER_TYPE_SELL' }, { id: '4', doneTime: new Date('2020-02-01T00:00:00.000Z'), type: 'ORDER_TYPE_SELL' }, { id: '8', doneTime: new Date('2020-02-01T00:00:00.000Z'), type: 'ORDER_TYPE_SELL' }, { id: '7', doneTime: new Date('2020-05-01T00:00:00.000Z'), type: 'ORDER_TYPE_BUY' }, { id: '5', doneTime: new Date('2020-06-01T00:00:00.000Z'), type: 'ORDER_TYPE_BUY' }, { id: '2', doneTime: new Date('2020-08-01T00:00:00.000Z'), type: 'ORDER_TYPE_SELL' }, { id: '3', doneTime: new Date('2020-09-01T00:00:00.000Z'), type: 'ORDER_TYPE_BUY' }, { id: '6', doneTime: new Date('2020-10-01T00:00:00.000Z'), type: 'ORDER_TYPE_BUY' }]);
  });

  /**
   * @test {MemoryHistoryStorage#orderSynchronizationFinished}
   */
  it('should return saved order synchronization status', () => {
    storage.orderSynchronizationFinished.should.be.false();
    storage.onHistoryOrdersSynchronized(1);
    storage.orderSynchronizationFinished.should.be.true();
  });

  /**
   * @test {MemoryHistoryStorage#dealSynchronizationFinished}
   */
  it('should return saved deal synchronization status', () => {
    storage._fileManager.updateDiskStorage = sandbox.stub();
    storage.dealSynchronizationFinished.should.be.false();
    storage.onDealsSynchronized(1);
    _sinon2.default.assert.calledOnce(storage._fileManager.updateDiskStorage);
    storage.dealSynchronizationFinished.should.be.true();
  });

  /**
   * @test {MemoryHistoryStorage#reset}
   */
  it('should reset storage', async () => {
    sandbox.stub(storage._fileManager, 'deleteStorageFromDisk');
    storage.onDealAdded('1:ps-mpa-1', { id: '1', time: new Date('2020-01-01T00:00:00.000Z'), type: 'DEAL_TYPE_SELL' });
    storage.onHistoryOrderAdded('1:ps-mpa-1', { id: '1', doneTime: new Date('2020-01-01T00:00:00.000Z'),
      type: 'ORDER_TYPE_SELL' });
    await storage.clear();
    _sinon2.default.assert.match(storage.deals, []);
    _sinon2.default.assert.match(storage.historyOrders, []);
    _sinon2.default.assert.calledOnce(storage._fileManager.deleteStorageFromDisk);
  });

  /**
   * @test {MemoryHistoryStorage#reset}
   */
  it('should record instance data from multiple streams', async () => {
    storage.onHistoryOrderAdded('1:ps-mpa-1', { id: '2', doneTime: new Date('2020-01-01T00:00:00.000Z') });
    _sinon2.default.assert.match((await storage.lastHistoryOrderTime(1)), new Date('2020-01-01T00:00:00.000Z'));
    storage.onHistoryOrderAdded('1:ps-mpa-2', { id: '3', doneTime: new Date('2020-01-02T00:00:00.000Z') });
    _sinon2.default.assert.match((await storage.lastHistoryOrderTime(1)), new Date('2020-01-02T00:00:00.000Z'));
    storage.onDealAdded('1:ps-mpa-1', { id: '2', time: new Date('2020-01-01T00:00:00.000Z') });
    _sinon2.default.assert.match((await storage.lastDealTime(1)), new Date('2020-01-01T00:00:00.000Z'));
    storage.onDealAdded('1:ps-mpa-2', { id: '3', time: new Date('2020-01-02T00:00:00.000Z') });
    _sinon2.default.assert.match((await storage.lastDealTime(1)), new Date('2020-01-02T00:00:00.000Z'));
  });
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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