'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _metatraderAccount = require('./metatraderAccount');

var _metatraderAccount2 = _interopRequireDefault(_metatraderAccount);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Exposes MetaTrader account API logic to the consumers
 */
class MetatraderAccountApi {

  /**
   * Constructs a MetaTrader account API instance
   * @param {MetatraderAccountClient} metatraderAccountClient MetaTrader account REST API client
   * @param {MetaApiWebsocketClient} metaApiWebsocketClient MetaApi websocket client
   * @param {ConnectionRegistry} connectionRegistry metatrader account connection registry
   * @param {ExpertAdvisorClient} expertAdvisorClient expert advisor REST API client
   * @param {HistoricalMarketDataClient} historicalMarketDataClient historical market data HTTP API client
   */
  constructor(metatraderAccountClient, metaApiWebsocketClient, connectionRegistry, expertAdvisorClient, historicalMarketDataClient) {
    this._metatraderAccountClient = metatraderAccountClient;
    this._metaApiWebsocketClient = metaApiWebsocketClient;
    this._connectionRegistry = connectionRegistry;
    this._expertAdvisorClient = expertAdvisorClient;
    this._historicalMarketDataClient = historicalMarketDataClient;
  }

  /**
   * Retrieves MetaTrader accounts
   * @param {AccountsFilter} accountsFilter optional filter
   * @return {Promise<Array<MetatraderAccount>>} promise resolving with an array of MetaTrader account entities
   */
  async getAccounts(accountsFilter) {
    let accounts = await this._metatraderAccountClient.getAccounts(accountsFilter);
    if (accounts.items) {
      accounts = accounts.items;
    }
    return accounts.map(a => new _metatraderAccount2.default(a, this._metatraderAccountClient, this._metaApiWebsocketClient, this._connectionRegistry, this._expertAdvisorClient, this._historicalMarketDataClient));
  }

  /**
   * Retrieves a MetaTrader account by id
   * @param {String} accountId MetaTrader account id
   * @return {Promise<MetatraderAccount>} promise resolving with MetaTrader account entity
   */
  async getAccount(accountId) {
    let account = await this._metatraderAccountClient.getAccount(accountId);
    return new _metatraderAccount2.default(account, this._metatraderAccountClient, this._metaApiWebsocketClient, this._connectionRegistry, this._expertAdvisorClient, this._historicalMarketDataClient);
  }

  /**
   * Retrieves a MetaTrader account by token
   * @return {Promise<MetatraderAccount>} promise resolving with MetaTrader account entity
   */
  async getAccountByToken() {
    let account = await this._metatraderAccountClient.getAccountByToken();
    return new _metatraderAccount2.default(account, this._metatraderAccountClient, this._metaApiWebsocketClient, this._connectionRegistry, this._expertAdvisorClient, this._historicalMarketDataClient);
  }

  /**
   * Creates a MetaTrader account
   * @param {NewMetatraderAccountDto} account MetaTrader account data
   * @return {Promise<MetatraderAccount>} promise resolving with MetaTrader account entity
   */
  async createAccount(account) {
    let id = await this._metatraderAccountClient.createAccount(account);
    return this.getAccount(id.id);
  }

}
exports.default = MetatraderAccountApi;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIi4uLy4uL2xpYi9tZXRhQXBpL21ldGF0cmFkZXJBY2NvdW50QXBpLmVzNiJdLCJuYW1lcyI6WyJNZXRhdHJhZGVyQWNjb3VudEFwaSIsImNvbnN0cnVjdG9yIiwibWV0YXRyYWRlckFjY291bnRDbGllbnQiLCJtZXRhQXBpV2Vic29ja2V0Q2xpZW50IiwiY29ubmVjdGlvblJlZ2lzdHJ5IiwiZXhwZXJ0QWR2aXNvckNsaWVudCIsImhpc3RvcmljYWxNYXJrZXREYXRhQ2xpZW50IiwiX21ldGF0cmFkZXJBY2NvdW50Q2xpZW50IiwiX21ldGFBcGlXZWJzb2NrZXRDbGllbnQiLCJfY29ubmVjdGlvblJlZ2lzdHJ5IiwiX2V4cGVydEFkdmlzb3JDbGllbnQiLCJfaGlzdG9yaWNhbE1hcmtldERhdGFDbGllbnQiLCJnZXRBY2NvdW50cyIsImFjY291bnRzRmlsdGVyIiwiYWNjb3VudHMiLCJpdGVtcyIsIm1hcCIsImEiLCJNZXRhdHJhZGVyQWNjb3VudCIsImdldEFjY291bnQiLCJhY2NvdW50SWQiLCJhY2NvdW50IiwiZ2V0QWNjb3VudEJ5VG9rZW4iLCJjcmVhdGVBY2NvdW50IiwiaWQiXSwibWFwcGluZ3MiOiJBQUFBOzs7Ozs7QUFFQTs7Ozs7O0FBRUE7OztBQUdlLE1BQU1BLG9CQUFOLENBQTJCOztBQUV4Qzs7Ozs7Ozs7QUFRQUMsY0FBWUMsdUJBQVosRUFBcUNDLHNCQUFyQyxFQUE2REMsa0JBQTdELEVBQWlGQyxtQkFBakYsRUFDRUMsMEJBREYsRUFDOEI7QUFDNUIsU0FBS0Msd0JBQUwsR0FBZ0NMLHVCQUFoQztBQUNBLFNBQUtNLHVCQUFMLEdBQStCTCxzQkFBL0I7QUFDQSxTQUFLTSxtQkFBTCxHQUEyQkwsa0JBQTNCO0FBQ0EsU0FBS00sb0JBQUwsR0FBNEJMLG1CQUE1QjtBQUNBLFNBQUtNLDJCQUFMLEdBQW1DTCwwQkFBbkM7QUFDRDs7QUFFRDs7Ozs7QUFLQSxRQUFNTSxXQUFOLENBQWtCQyxjQUFsQixFQUFrQztBQUNoQyxRQUFJQyxXQUFXLE1BQU0sS0FBS1Asd0JBQUwsQ0FBOEJLLFdBQTlCLENBQTBDQyxjQUExQyxDQUFyQjtBQUNBLFFBQUlDLFNBQVNDLEtBQWIsRUFBb0I7QUFDbEJELGlCQUFXQSxTQUFTQyxLQUFwQjtBQUNEO0FBQ0QsV0FBT0QsU0FBU0UsR0FBVCxDQUFhQyxLQUFLLElBQUlDLDJCQUFKLENBQXNCRCxDQUF0QixFQUF5QixLQUFLVix3QkFBOUIsRUFBd0QsS0FBS0MsdUJBQTdELEVBQ3ZCLEtBQUtDLG1CQURrQixFQUNHLEtBQUtDLG9CQURSLEVBQzhCLEtBQUtDLDJCQURuQyxDQUFsQixDQUFQO0FBRUQ7O0FBRUQ7Ozs7O0FBS0EsUUFBTVEsVUFBTixDQUFpQkMsU0FBakIsRUFBNEI7QUFDMUIsUUFBSUMsVUFBVSxNQUFNLEtBQUtkLHdCQUFMLENBQThCWSxVQUE5QixDQUF5Q0MsU0FBekMsQ0FBcEI7QUFDQSxXQUFPLElBQUlGLDJCQUFKLENBQXNCRyxPQUF0QixFQUErQixLQUFLZCx3QkFBcEMsRUFBOEQsS0FBS0MsdUJBQW5FLEVBQ0wsS0FBS0MsbUJBREEsRUFDc0IsS0FBS0Msb0JBRDNCLEVBQ2lELEtBQUtDLDJCQUR0RCxDQUFQO0FBRUQ7O0FBRUQ7Ozs7QUFJQSxRQUFNVyxpQkFBTixHQUEwQjtBQUN4QixRQUFJRCxVQUFVLE1BQU0sS0FBS2Qsd0JBQUwsQ0FBOEJlLGlCQUE5QixFQUFwQjtBQUNBLFdBQU8sSUFBSUosMkJBQUosQ0FBc0JHLE9BQXRCLEVBQStCLEtBQUtkLHdCQUFwQyxFQUE4RCxLQUFLQyx1QkFBbkUsRUFDTCxLQUFLQyxtQkFEQSxFQUNxQixLQUFLQyxvQkFEMUIsRUFDZ0QsS0FBS0MsMkJBRHJELENBQVA7QUFFRDs7QUFFRDs7Ozs7QUFLQSxRQUFNWSxhQUFOLENBQW9CRixPQUFwQixFQUE2QjtBQUMzQixRQUFJRyxLQUFLLE1BQU0sS0FBS2pCLHdCQUFMLENBQThCZ0IsYUFBOUIsQ0FBNENGLE9BQTVDLENBQWY7QUFDQSxXQUFPLEtBQUtGLFVBQUwsQ0FBZ0JLLEdBQUdBLEVBQW5CLENBQVA7QUFDRDs7QUE5RHVDO2tCQUFyQnhCLG9CIiwiZmlsZSI6Im1ldGF0cmFkZXJBY2NvdW50QXBpLmpzIiwic291cmNlc0NvbnRlbnQiOlsiJ3VzZSBzdHJpY3QnO1xuXG5pbXBvcnQgTWV0YXRyYWRlckFjY291bnQgZnJvbSAnLi9tZXRhdHJhZGVyQWNjb3VudCc7XG5cbi8qKlxuICogRXhwb3NlcyBNZXRhVHJhZGVyIGFjY291bnQgQVBJIGxvZ2ljIHRvIHRoZSBjb25zdW1lcnNcbiAqL1xuZXhwb3J0IGRlZmF1bHQgY2xhc3MgTWV0YXRyYWRlckFjY291bnRBcGkge1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3RzIGEgTWV0YVRyYWRlciBhY2NvdW50IEFQSSBpbnN0YW5jZVxuICAgKiBAcGFyYW0ge01ldGF0cmFkZXJBY2NvdW50Q2xpZW50fSBtZXRhdHJhZGVyQWNjb3VudENsaWVudCBNZXRhVHJhZGVyIGFjY291bnQgUkVTVCBBUEkgY2xpZW50XG4gICAqIEBwYXJhbSB7TWV0YUFwaVdlYnNvY2tldENsaWVudH0gbWV0YUFwaVdlYnNvY2tldENsaWVudCBNZXRhQXBpIHdlYnNvY2tldCBjbGllbnRcbiAgICogQHBhcmFtIHtDb25uZWN0aW9uUmVnaXN0cnl9IGNvbm5lY3Rpb25SZWdpc3RyeSBtZXRhdHJhZGVyIGFjY291bnQgY29ubmVjdGlvbiByZWdpc3RyeVxuICAgKiBAcGFyYW0ge0V4cGVydEFkdmlzb3JDbGllbnR9IGV4cGVydEFkdmlzb3JDbGllbnQgZXhwZXJ0IGFkdmlzb3IgUkVTVCBBUEkgY2xpZW50XG4gICAqIEBwYXJhbSB7SGlzdG9yaWNhbE1hcmtldERhdGFDbGllbnR9IGhpc3RvcmljYWxNYXJrZXREYXRhQ2xpZW50IGhpc3RvcmljYWwgbWFya2V0IGRhdGEgSFRUUCBBUEkgY2xpZW50XG4gICAqL1xuICBjb25zdHJ1Y3RvcihtZXRhdHJhZGVyQWNjb3VudENsaWVudCwgbWV0YUFwaVdlYnNvY2tldENsaWVudCwgY29ubmVjdGlvblJlZ2lzdHJ5LCBleHBlcnRBZHZpc29yQ2xpZW50LCBcbiAgICBoaXN0b3JpY2FsTWFya2V0RGF0YUNsaWVudCkge1xuICAgIHRoaXMuX21ldGF0cmFkZXJBY2NvdW50Q2xpZW50ID0gbWV0YXRyYWRlckFjY291bnRDbGllbnQ7XG4gICAgdGhpcy5fbWV0YUFwaVdlYnNvY2tldENsaWVudCA9IG1ldGFBcGlXZWJzb2NrZXRDbGllbnQ7XG4gICAgdGhpcy5fY29ubmVjdGlvblJlZ2lzdHJ5ID0gY29ubmVjdGlvblJlZ2lzdHJ5O1xuICAgIHRoaXMuX2V4cGVydEFkdmlzb3JDbGllbnQgPSBleHBlcnRBZHZpc29yQ2xpZW50O1xuICAgIHRoaXMuX2hpc3RvcmljYWxNYXJrZXREYXRhQ2xpZW50ID0gaGlzdG9yaWNhbE1hcmtldERhdGFDbGllbnQ7XG4gIH1cblxuICAvKipcbiAgICogUmV0cmlldmVzIE1ldGFUcmFkZXIgYWNjb3VudHNcbiAgICogQHBhcmFtIHtBY2NvdW50c0ZpbHRlcn0gYWNjb3VudHNGaWx0ZXIgb3B0aW9uYWwgZmlsdGVyXG4gICAqIEByZXR1cm4ge1Byb21pc2U8QXJyYXk8TWV0YXRyYWRlckFjY291bnQ+Pn0gcHJvbWlzZSByZXNvbHZpbmcgd2l0aCBhbiBhcnJheSBvZiBNZXRhVHJhZGVyIGFjY291bnQgZW50aXRpZXNcbiAgICovXG4gIGFzeW5jIGdldEFjY291bnRzKGFjY291bnRzRmlsdGVyKSB7XG4gICAgbGV0IGFjY291bnRzID0gYXdhaXQgdGhpcy5fbWV0YXRyYWRlckFjY291bnRDbGllbnQuZ2V0QWNjb3VudHMoYWNjb3VudHNGaWx0ZXIpO1xuICAgIGlmIChhY2NvdW50cy5pdGVtcykge1xuICAgICAgYWNjb3VudHMgPSBhY2NvdW50cy5pdGVtcztcbiAgICB9XG4gICAgcmV0dXJuIGFjY291bnRzLm1hcChhID0+IG5ldyBNZXRhdHJhZGVyQWNjb3VudChhLCB0aGlzLl9tZXRhdHJhZGVyQWNjb3VudENsaWVudCwgdGhpcy5fbWV0YUFwaVdlYnNvY2tldENsaWVudCwgXG4gICAgICB0aGlzLl9jb25uZWN0aW9uUmVnaXN0cnksIHRoaXMuX2V4cGVydEFkdmlzb3JDbGllbnQsIHRoaXMuX2hpc3RvcmljYWxNYXJrZXREYXRhQ2xpZW50KSk7XG4gIH1cblxuICAvKipcbiAgICogUmV0cmlldmVzIGEgTWV0YVRyYWRlciBhY2NvdW50IGJ5IGlkXG4gICAqIEBwYXJhbSB7U3RyaW5nfSBhY2NvdW50SWQgTWV0YVRyYWRlciBhY2NvdW50IGlkXG4gICAqIEByZXR1cm4ge1Byb21pc2U8TWV0YXRyYWRlckFjY291bnQ+fSBwcm9taXNlIHJlc29sdmluZyB3aXRoIE1ldGFUcmFkZXIgYWNjb3VudCBlbnRpdHlcbiAgICovXG4gIGFzeW5jIGdldEFjY291bnQoYWNjb3VudElkKSB7XG4gICAgbGV0IGFjY291bnQgPSBhd2FpdCB0aGlzLl9tZXRhdHJhZGVyQWNjb3VudENsaWVudC5nZXRBY2NvdW50KGFjY291bnRJZCk7XG4gICAgcmV0dXJuIG5ldyBNZXRhdHJhZGVyQWNjb3VudChhY2NvdW50LCB0aGlzLl9tZXRhdHJhZGVyQWNjb3VudENsaWVudCwgdGhpcy5fbWV0YUFwaVdlYnNvY2tldENsaWVudCwgXG4gICAgICB0aGlzLl9jb25uZWN0aW9uUmVnaXN0cnksICB0aGlzLl9leHBlcnRBZHZpc29yQ2xpZW50LCB0aGlzLl9oaXN0b3JpY2FsTWFya2V0RGF0YUNsaWVudCk7XG4gIH1cblxuICAvKipcbiAgICogUmV0cmlldmVzIGEgTWV0YVRyYWRlciBhY2NvdW50IGJ5IHRva2VuXG4gICAqIEByZXR1cm4ge1Byb21pc2U8TWV0YXRyYWRlckFjY291bnQ+fSBwcm9taXNlIHJlc29sdmluZyB3aXRoIE1ldGFUcmFkZXIgYWNjb3VudCBlbnRpdHlcbiAgICovXG4gIGFzeW5jIGdldEFjY291bnRCeVRva2VuKCkge1xuICAgIGxldCBhY2NvdW50ID0gYXdhaXQgdGhpcy5fbWV0YXRyYWRlckFjY291bnRDbGllbnQuZ2V0QWNjb3VudEJ5VG9rZW4oKTtcbiAgICByZXR1cm4gbmV3IE1ldGF0cmFkZXJBY2NvdW50KGFjY291bnQsIHRoaXMuX21ldGF0cmFkZXJBY2NvdW50Q2xpZW50LCB0aGlzLl9tZXRhQXBpV2Vic29ja2V0Q2xpZW50LCBcbiAgICAgIHRoaXMuX2Nvbm5lY3Rpb25SZWdpc3RyeSwgdGhpcy5fZXhwZXJ0QWR2aXNvckNsaWVudCwgdGhpcy5faGlzdG9yaWNhbE1hcmtldERhdGFDbGllbnQpO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBNZXRhVHJhZGVyIGFjY291bnRcbiAgICogQHBhcmFtIHtOZXdNZXRhdHJhZGVyQWNjb3VudER0b30gYWNjb3VudCBNZXRhVHJhZGVyIGFjY291bnQgZGF0YVxuICAgKiBAcmV0dXJuIHtQcm9taXNlPE1ldGF0cmFkZXJBY2NvdW50Pn0gcHJvbWlzZSByZXNvbHZpbmcgd2l0aCBNZXRhVHJhZGVyIGFjY291bnQgZW50aXR5XG4gICAqL1xuICBhc3luYyBjcmVhdGVBY2NvdW50KGFjY291bnQpIHtcbiAgICBsZXQgaWQgPSBhd2FpdCB0aGlzLl9tZXRhdHJhZGVyQWNjb3VudENsaWVudC5jcmVhdGVBY2NvdW50KGFjY291bnQpO1xuICAgIHJldHVybiB0aGlzLmdldEFjY291bnQoaWQuaWQpO1xuICB9XG5cbn1cbiJdfQ==