'use strict';

/**
 * Defines interface for a terminal listener class. You can extend your trading application implementation from
 * this class.
 * Unlike low-level SynchronizationListener, when you use TerminalListener, the SDK will remove race conditions and most
 * events duplicates present on the transport layer. There is also a warranty that TeminalListener listener methods
 * will be invoked in sequence. No new event will be delivered until previous event have finished processing.
 * So that it is much easier to create trading applicaitions using TerminalListener, especially if the application is a
 * complex one.
 * Please note that some event listener methods can still receive event duplicates, so that they must be idempotent.
 * The terminal lifecycle starts with onConnected event, followed by initial synchronization, followed by data updates.
 * During initial synchronization the SDK will invoke the following methods in sequence: onSynchronizationStarted,
 * onSymbolSpecificationUpdated (invoked both for new and updated symbol specifications, can be skipped if
 * specifications data has not changed), onSymbolSpecificationRemoved, onQuote, onAccountInformationUpdated,
 * onPositionsReplaced (can be skipped if position data has not changed), onPositionsSynchronized,
 * onPendingOrdersReplaced (can be skipped if pending order data has not changed), onPendingOrdersSynchronized,
 * onHistoryOrderAdded, onHistoryOrdersSynchronized, onDealAdded, onDealsSynchronized.
 * Data update events are onPositionUpdated (invoked both for new and updated positions), onPositionRemoved,
 * onPendingOrderUpdated (invoked both for new and updated pending orders), onPendingOrderCompleted,
 * onHistoryOrderAdded, onDealAdded, onSymbolSpecificationUpdated (invoked both for new and updated symbol
 * specifications), onSymbolSpecificationRemoved, onQuote, onCandle, onTick, onBook.
 * There are also status events available such as onBrokerConnectionStatus, onDisconnected,
 * onSubscriptionDowngraded.
 */

Object.defineProperty(exports, "__esModule", {
  value: true
});
class TerminalListener {

  /**
   * Returns MetaApiConnection instance
   * @return {MetaApiConnection} MetaApiConnection instance
   */
  get connection() {
    return this._connection;
  }

  /**
   * Sets MetaApiConnection instance
   * @param {MetaApiConnection} connection MetaApiConnection instance
   */
  set connection(connection) {
    this._connection = connection;
  }

  /**
   * Invoked when connection to MetaTrader terminal established
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onConnected() {}

  /**
   * Invoked when connection to MetaTrader terminal terminated
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onDisconnected() {}

  /**
   * Invoked when broker connection satus have changed
   * @param {Boolean} connected is MetaTrader terminal is connected to broker
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onBrokerConnectionStatus(connected) {}

  /**
   * Invoked when MetaTrader terminal state synchronization is started
   * @param {Boolean} specificationsUpdated whether specifications are going to be updated during synchronization
   * @param {Boolean} positionsUpdated whether positions are going to be updated during synchronization
   * @param {Boolean} ordersUpdated whether orders are going to be updated during synchronization
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onSynchronizationStarted(specificationsUpdated, positionsUpdated, ordersUpdated) {}

  /**
   * Invoked when MetaTrader account information is updated
   * @param {MetatraderAccountInformation} accountInformation updated MetaTrader account information
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onAccountInformationUpdated(accountInformation) {}

  /**
   * Invoked when the positions are replaced as a result of initial terminal state synchronization. This method
   * will be invoked only if server thinks the data was updated, otherwise invocation can be skipped
   * @param {Array<MetatraderPosition>} positions updated array of positions
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onPositionsReplaced(positions) {}

  /**
   * Invoked when position synchronization fnished to indicate progress of an initial terminal state synchronization
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onPositionsSynchronized() {}

  /**
   * Invoked when MetaTrader position is updated
   * @param {MetatraderPosition} position updated MetaTrader position
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onPositionUpdated(position) {}

  /**
   * Invoked when MetaTrader position is removed
   * @param {String} positionId removed MetaTrader position id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onPositionRemoved(positionId) {}

  /**
   * Invoked when the pending orders are replaced as a result of initial terminal state synchronization. This method
   * will be invoked only if server thinks the data was updated, otherwise invocation can be skipped
   * @param {Array<MetatraderOrder>} orders updated array of pending orders
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onPendingOrdersReplaced(orders) {}

  /**
   * Invoked when MetaTrader pending order is updated
   * @param {MetatraderOrder} order updated MetaTrader pending order
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onPendingOrderUpdated(order) {}

  /**
   * Invoked when MetaTrader pending order is completed (executed or canceled)
   * @param {String} orderId completed MetaTrader pending order id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onPendingOrderCompleted(orderId) {}

  /**
   * Invoked when pending order synchronization fnished to indicate progress of an initial terminal state
   * synchronization
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onPendingOrdersSynchronized() {}

  /**
   * Invoked when a new MetaTrader history order is added
   * @param {MetatraderOrder} historyOrder new MetaTrader history order
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onHistoryOrderAdded(historyOrder) {}

  /**
   * Invoked when a synchronization of history orders on a MetaTrader account have finished to indicate progress of an
   * initial terminal state synchronization
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onHistoryOrdersSynchronized() {}

  /**
   * Invoked when a new MetaTrader history deal is added
   * @param {MetatraderDeal} deal new MetaTrader history deal
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onDealAdded(deal) {}

  /**
   * Invoked when a synchronization of history deals on a MetaTrader account have finished to indicate progress of an
   * initial terminal state synchronization
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onDealsSynchronized() {}

  /**
   * Invoked when a symbol specification was updated
   * @param {MetatraderSymbolSpecification} specification updated MetaTrader symbol specification
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onSymbolSpecificationUpdated(specification) {}

  /**
   * Invoked when a symbol specification was removed
   * @param {String} symbol removed symbol
   * @returns {Promise} promise which resolves when the asynchronous event is processed
   */
  async onSymbolSpecificationRemoved(symbol) {}

  /**
   * Invoked when a symbol price was updated
   * @param {MetatraderSymbolPrice} quote updated MetaTrader symbol quote
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onQuote(quote) {}

  /**
   * Invoked when symbol candles were updated
   * @param {MetatraderCandle} candle updated MetaTrader symbol candle
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onCandle(candles) {}

  /**
   * Invoked when symbol ticks were updated
   * @param {MetatraderTick} tick updated MetaTrader symbol tick
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onTick(tick) {}

  /**
   * Invoked when order books were updated
   * @param {MetatraderBook} book updated MetaTrader order book
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onBook(book) {}

  /**
   * Invoked when subscription downgrade has occurred
   * @param {string} symbol symbol to update subscriptions for
   * @param {Array<MarketDataSubscription>} updates array of market data subscription to update
   * @param {Array<MarketDataUnsubscription>} unsubscriptions array of subscriptions to cancel
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */
  async onSubscriptionDowngraded(symbol, updates, unsubscriptions) {}

}
exports.default = TerminalListener;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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